<?php
namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\User;
use Illuminate\Http\Request;

class ClientController extends Controller
{
    public function index(Request $request)
    {
        $clients = User::clients()
            ->when($request->search, fn($q, $s) => $q->search($s))
            ->when($request->status === 'active', fn($q) => $q->active())
            ->when($request->status === 'inactive', fn($q) => $q->where('is_active', false))
            ->withCount(['services', 'domains', 'invoices'])
            ->latest()
            ->paginate(25);

        return view('admin.clients.index', compact('clients'));
    }

    public function create()
    {
        return view('admin.clients.create');
    }

    public function store(Request $request)
    {
        $data = $request->validate([
            'first_name' => 'required|max:100', 'last_name' => 'required|max:100',
            'email' => 'required|email|unique:users', 'password' => 'required|min:8',
            'company_name' => 'nullable|max:200', 'phone' => 'nullable|max:30',
            'address1' => 'nullable|max:255', 'city' => 'nullable|max:100',
            'state' => 'nullable|max:100', 'postcode' => 'nullable|max:20',
            'country' => 'required|size:2',
        ]);
        $data['role'] = 'client';
        User::create($data);
        return redirect()->route('admin.clients.index')->with('success', 'Client created successfully.');
    }

    public function show(User $client)
    {
        $client->load(['services.product', 'invoices', 'domains', 'tickets']);
        return view('admin.clients.show', compact('client'));
    }

    public function edit(User $client)
    {
        return view('admin.clients.edit', compact('client'));
    }

    public function update(Request $request, User $client)
    {
        $data = $request->validate([
            'first_name' => 'required|max:100', 'last_name' => 'required|max:100',
            'email' => 'required|email|unique:users,email,' . $client->id,
            'company_name' => 'nullable|max:200', 'phone' => 'nullable|max:30',
            'address1' => 'nullable|max:255', 'city' => 'nullable|max:100',
            'state' => 'nullable|max:100', 'postcode' => 'nullable|max:20',
            'country' => 'required|size:2', 'notes' => 'nullable|max:1000',
        ]);
        if ($request->filled('password')) {
            $data['password'] = bcrypt($request->password);
        }
        $client->update($data);
        return redirect()->route('admin.clients.show', $client)->with('success', 'Client updated.');
    }

    public function toggleStatus(User $client)
    {
        $client->update(['is_active' => !$client->is_active]);
        return back()->with('success', 'Client status updated.');
    }

    public function addCredit(Request $request, User $client)
    {
        $request->validate(['amount' => 'required|numeric|min:0.01', 'description' => 'required|max:500']);
        $client->addCredit($request->amount, $request->description, auth()->id());
        return back()->with('success', "Credit of {$request->amount} added.");
    }
}
