<?php
namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\{Invoice, InvoiceItem, User, Transaction};
use Illuminate\Http\Request;

class InvoiceController extends Controller
{
    public function index(Request $request)
    {
        $invoices = Invoice::with('user')
            ->when($request->status, fn($q, $s) => $q->where('status', $s))
            ->when($request->search, fn($q, $s) => $q->whereHas('user', fn($u) => $u->search($s)))
            ->latest()
            ->paginate(25);
        return view('admin.invoices.index', compact('invoices'));
    }

    public function create()
    {
        $clients = User::clients()->active()->orderBy('first_name')->get();
        return view('admin.invoices.create', compact('clients'));
    }

    public function store(Request $request)
    {
        $request->validate([
            'user_id' => 'required|exists:users,id',
            'date_due' => 'required|date',
            'items' => 'required|array|min:1',
            'items.*.description' => 'required|max:500',
            'items.*.amount' => 'required|numeric',
        ]);

        $invoice = Invoice::create([
            'user_id' => $request->user_id,
            'status' => 'unpaid',
            'date_issued' => now(),
            'date_due' => $request->date_due,
            'payment_method' => $request->payment_method,
            'notes' => $request->notes,
        ]);

        foreach ($request->items as $item) {
            $invoice->items()->create([
                'type' => $item['type'] ?? 'other',
                'description' => $item['description'],
                'amount' => $item['amount'],
            ]);
        }

        $invoice->recalculate();
        return redirect()->route('admin.invoices.show', $invoice)->with('success', 'Invoice created.');
    }

    public function show(Invoice $invoice)
    {
        $invoice->load(['user', 'items.service', 'transactions']);
        return view('admin.invoices.show', compact('invoice'));
    }

    public function markPaid(Request $request, Invoice $invoice)
    {
        $invoice->markPaid($request->payment_method ?? 'manual');
        Transaction::create([
            'user_id' => $invoice->user_id, 'invoice_id' => $invoice->id,
            'gateway' => 'manual', 'type' => 'payment',
            'amount' => $invoice->total, 'currency' => $invoice->currency,
            'status' => 'completed', 'description' => 'Manual payment recorded',
        ]);
        return back()->with('success', 'Invoice marked as paid.');
    }

    public function cancelInvoice(Invoice $invoice)
    {
        $invoice->update(['status' => 'cancelled']);
        return back()->with('success', 'Invoice cancelled.');
    }

    public function downloadPdf(Invoice $invoice)
    {
        $invoice->load(['user', 'items']);
        $pdf = \Barryvdh\DomPDF\Facade\Pdf::loadView('admin.invoices.pdf', compact('invoice'));
        return $pdf->download("Invoice-{$invoice->invoice_number}.pdf");
    }
}
