<?php
namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\{Product, ProductGroup, ProductPricing, ServerGroup};
use Illuminate\Http\Request;
use Illuminate\Support\Str;

class ProductController extends Controller
{
    public function index(Request $request)
    {
        $products = Product::with('group')
            ->when($request->group, fn($q, $g) => $q->where('product_group_id', $g))
            ->when($request->type, fn($q, $t) => $q->where('type', $t))
            ->withCount('services')
            ->orderBy('sort_order')
            ->paginate(25);

        $groups = ProductGroup::orderBy('sort_order')->get();
        return view('admin.products.index', compact('products', 'groups'));
    }

    public function create()
    {
        $groups = ProductGroup::orderBy('sort_order')->get();
        $serverGroups = ServerGroup::with('servers')->get();
        return view('admin.products.create', compact('groups', 'serverGroups'));
    }

    public function store(Request $request)
    {
        $data = $request->validate([
            'product_group_id' => 'required|exists:product_groups,id',
            'name' => 'required|max:200', 'type' => 'required|in:hosting,reseller,vps,dedicated,domain,addon,other',
            'description' => 'nullable', 'module' => 'nullable|max:100',
            'auto_setup' => 'boolean', 'auto_setup_on' => 'in:payment,order,manual',
            'require_domain' => 'boolean',
        ]);
        $data['slug'] = Str::slug($data['name']);

        $product = Product::create($data);

        // Save pricing
        if ($request->has('pricing')) {
            foreach ($request->pricing as $cycle => $price) {
                if ($price !== null && $price !== '') {
                    ProductPricing::create([
                        'product_id' => $product->id, 'billing_cycle' => $cycle,
                        'price' => $price, 'setup_fee' => $request->input("setup_fee.{$cycle}", 0),
                    ]);
                }
            }
        }

        if ($request->has('server_groups')) {
            $product->serverGroups()->sync($request->server_groups);
        }

        return redirect()->route('admin.products.index')->with('success', 'Product created.');
    }

    public function edit(Product $product)
    {
        $product->load('pricing', 'serverGroups');
        $groups = ProductGroup::orderBy('sort_order')->get();
        $serverGroups = ServerGroup::with('servers')->get();
        return view('admin.products.edit', compact('product', 'groups', 'serverGroups'));
    }

    public function update(Request $request, Product $product)
    {
        $data = $request->validate([
            'product_group_id' => 'required|exists:product_groups,id',
            'name' => 'required|max:200', 'type' => 'required',
            'description' => 'nullable', 'module' => 'nullable|max:100',
        ]);
        $product->update($data);

        // Update pricing
        $product->pricing()->delete();
        if ($request->has('pricing')) {
            foreach ($request->pricing as $cycle => $price) {
                if ($price !== null && $price !== '') {
                    ProductPricing::create([
                        'product_id' => $product->id, 'billing_cycle' => $cycle,
                        'price' => $price, 'setup_fee' => $request->input("setup_fee.{$cycle}", 0),
                    ]);
                }
            }
        }

        return redirect()->route('admin.products.edit', $product)->with('success', 'Product updated.');
    }

    public function destroy(Product $product)
    {
        if ($product->services()->exists()) {
            return back()->with('error', 'Cannot delete product with active services.');
        }
        $product->delete();
        return redirect()->route('admin.products.index')->with('success', 'Product deleted.');
    }
}
