<?php
namespace App\Http\Controllers\Auth;

use App\Http\Controllers\Controller;
use App\Models\{User, ActivityLog};
use Illuminate\Http\Request;
use Illuminate\Support\Facades\{Auth, Hash, Password};

class AuthController extends Controller
{
    public function showLogin() { return view('auth.login'); }

    public function login(Request $request)
    {
        $credentials = $request->validate(['email' => 'required|email', 'password' => 'required']);

        if (Auth::attempt($credentials, $request->boolean('remember'))) {
            $user = Auth::user();
            $user->update(['last_login_at' => now(), 'last_login_ip' => $request->ip()]);
            ActivityLog::create([
                'user_id' => $user->id, 'type' => 'login',
                'description' => 'User logged in', 'ip_address' => $request->ip(),
            ]);
            return $user->isAdmin() ? redirect()->route('admin.dashboard') : redirect()->route('client.dashboard');
        }
        return back()->withErrors(['email' => 'Invalid credentials.'])->onlyInput('email');
    }

    public function showRegister() { return view('auth.register'); }

    public function register(Request $request)
    {
        $validated = $request->validate([
            'first_name' => 'required|string|max:255', 'last_name' => 'required|string|max:255',
            'email' => 'required|email|unique:users', 'password' => 'required|confirmed|min:8',
            'company_name' => 'nullable|string', 'phone' => 'nullable|string',
            'address1' => 'nullable|string', 'city' => 'nullable|string',
            'state' => 'nullable|string', 'postcode' => 'nullable|string',
            'country' => 'required|string|size:2',
        ]);
        $validated['password'] = Hash::make($validated['password']);
        $validated['role'] = 'client';
        Auth::login(User::create($validated));
        return redirect()->route('client.dashboard');
    }

    public function logout(Request $request)
    {
        Auth::logout();
        $request->session()->invalidate();
        $request->session()->regenerateToken();
        return redirect()->route('login');
    }

    public function showForgotPassword() { return view('auth.forgot-password'); }
    public function sendResetLink(Request $request) {
        $request->validate(['email' => 'required|email']);
        Password::sendResetLink($request->only('email'));
        return back()->with('status', 'Reset link sent if email exists.');
    }
    public function showResetPassword(string $token) { return view('auth.reset-password', compact('token')); }
    public function resetPassword(Request $request) {
        $request->validate(['token' => 'required', 'email' => 'required|email', 'password' => 'required|confirmed|min:8']);
        Password::reset($request->only('email', 'password', 'password_confirmation', 'token'),
            fn(User $user, string $password) => $user->update(['password' => Hash::make($password)]));
        return redirect()->route('login')->with('status', 'Password reset.');
    }
}
