<?php
namespace App\Http\Controllers\Client;

use App\Http\Controllers\Controller;
use App\Models\{Product, ProductGroup, Order, Service, Invoice, InvoiceItem};
use Illuminate\Http\Request;

class OrderController extends Controller
{
    public function index()
    {
        $groups = ProductGroup::with(['activeProducts.pricing'])->where('is_hidden', false)->orderBy('sort_order')->get();
        return view('client.order.index', compact('groups'));
    }

    public function configure(Product $product)
    {
        $product->load('pricing');
        return view('client.order.configure', compact('product'));
    }

    public function store(Request $request)
    {
        $request->validate([
            'product_id' => 'required|exists:products,id',
            'billing_cycle' => 'required',
            'domain' => 'nullable|max:255',
        ]);

        $product = Product::findOrFail($request->product_id);
        $pricing = $product->pricing()->where('billing_cycle', $request->billing_cycle)->firstOrFail();

        // Create invoice
        $invoice = Invoice::create([
            'user_id' => auth()->id(), 'status' => 'unpaid',
            'date_issued' => now(), 'date_due' => now()->addDays(7),
            'subtotal' => $pricing->price + $pricing->setup_fee,
            'total' => $pricing->price + $pricing->setup_fee,
            'balance' => $pricing->price + $pricing->setup_fee,
        ]);

        // Create order
        $order = Order::create([
            'user_id' => auth()->id(), 'invoice_id' => $invoice->id,
            'status' => 'pending', 'amount' => $invoice->total,
            'ip_address' => $request->ip(),
        ]);

        // Create service
        $service = Service::create([
            'user_id' => auth()->id(), 'order_id' => $order->id,
            'product_id' => $product->id, 'domain' => $request->domain,
            'status' => 'pending', 'billing_cycle' => $request->billing_cycle,
            'recurring_amount' => $pricing->price,
        ]);

        // Add invoice items
        $invoice->items()->create([
            'service_id' => $service->id, 'type' => 'hosting',
            'description' => $product->name . ' - ' . ucfirst(str_replace('_', ' ', $request->billing_cycle)),
            'amount' => $pricing->price,
        ]);
        if ($pricing->setup_fee > 0) {
            $invoice->items()->create([
                'service_id' => $service->id, 'type' => 'setup',
                'description' => $product->name . ' - Setup Fee',
                'amount' => $pricing->setup_fee,
            ]);
        }

        return redirect()->route('client.invoices.show', $invoice)->with('success', 'Order placed! Please complete payment.');
    }
}
