<?php
namespace App\Http\Controllers\Client;

use App\Http\Controllers\Controller;
use App\Models\{Ticket, TicketDepartment, KbCategory, KbArticle};
use Illuminate\Http\Request;

class SupportController extends Controller
{
    public function index()
    {
        $tickets = auth()->user()->tickets()
            ->with('department')
            ->latest()
            ->paginate(20);
        return view('client.support.index', compact('tickets'));
    }

    public function create()
    {
        $departments = TicketDepartment::where('is_active', true)->orderBy('sort_order')->get();
        $services = auth()->user()->activeServices()->with('product')->get();
        return view('client.support.create', compact('departments', 'services'));
    }

    public function store(Request $request)
    {
        $validated = $request->validate([
            'department_id' => 'required|exists:ticket_departments,id',
            'subject' => 'required|string|max:255',
            'message' => 'required|string',
            'priority' => 'required|in:low,medium,high,urgent',
            'service_id' => 'nullable|exists:services,id',
        ]);

        $ticket = Ticket::create([
            'ticket_number' => Ticket::generateNumber(),
            'user_id' => auth()->id(),
            'department_id' => $validated['department_id'],
            'subject' => $validated['subject'],
            'priority' => $validated['priority'],
            'service_id' => $validated['service_id'] ?? null,
            'status' => 'open',
            'last_reply_at' => now(),
        ]);

        $ticket->replies()->create([
            'user_id' => auth()->id(),
            'message' => $validated['message'],
            'is_admin' => false,
        ]);

        return redirect()->route('client.support.show', $ticket)->with('success', 'Ticket created.');
    }

    public function show(Ticket $ticket)
    {
        abort_if($ticket->user_id !== auth()->id(), 403);
        $ticket->load(['replies.user', 'department']);
        return view('client.support.show', compact('ticket'));
    }

    public function reply(Request $request, Ticket $ticket)
    {
        abort_if($ticket->user_id !== auth()->id(), 403);
        $request->validate(['message' => 'required|string']);

        $ticket->replies()->create([
            'user_id' => auth()->id(),
            'message' => $request->message,
            'is_admin' => false,
        ]);
        $ticket->update(['status' => 'customer_reply', 'last_reply_at' => now()]);

        return back()->with('success', 'Reply sent.');
    }

    public function close(Ticket $ticket)
    {
        abort_if($ticket->user_id !== auth()->id(), 403);
        $ticket->update(['status' => 'closed']);
        return back()->with('success', 'Ticket closed.');
    }

    public function knowledgebase()
    {
        $categories = KbCategory::with('articles')->where('is_active', true)->orderBy('sort_order')->get();
        return view('client.support.knowledgebase', compact('categories'));
    }

    public function kbArticle(string $slug)
    {
        $article = KbArticle::where('slug', $slug)->where('is_active', true)->firstOrFail();
        $article->increment('views');
        return view('client.support.kb-article', compact('article'));
    }
}
