<?php
namespace App\Http\Controllers\Client;

use App\Http\Controllers\Controller;
use App\Models\{Ticket, Department};
use Illuminate\Http\Request;

class TicketController extends Controller
{
    public function index()
    {
        $tickets = auth()->user()->tickets()->with('department')->latest()->paginate(20);
        return view('client.tickets.index', compact('tickets'));
    }

    public function create()
    {
        $departments = Department::where('is_hidden', false)->orderBy('sort_order')->get();
        $services = auth()->user()->services()->with('product')->where('status', 'active')->get();
        return view('client.tickets.create', compact('departments', 'services'));
    }

    public function store(Request $request)
    {
        $data = $request->validate([
            'department_id' => 'required|exists:departments,id',
            'subject' => 'required|max:500', 'message' => 'required',
            'priority' => 'in:low,medium,high', 'service_id' => 'nullable|exists:services,id',
        ]);

        $ticket = auth()->user()->tickets()->create([
            'department_id' => $data['department_id'],
            'subject' => $data['subject'],
            'priority' => $data['priority'] ?? 'medium',
            'service_id' => $data['service_id'] ?? null,
            'status' => 'open',
        ]);

        $ticket->replies()->create([
            'user_id' => auth()->id(), 'message' => $data['message'],
            'ip_address' => $request->ip(),
        ]);

        return redirect()->route('client.tickets.show', $ticket)->with('success', 'Ticket opened.');
    }

    public function show(Ticket $ticket)
    {
        abort_if($ticket->user_id !== auth()->id(), 403);
        $ticket->load(['replies.user', 'department']);
        return view('client.tickets.show', compact('ticket'));
    }

    public function reply(Request $request, Ticket $ticket)
    {
        abort_if($ticket->user_id !== auth()->id(), 403);
        $request->validate(['message' => 'required']);
        $ticket->replies()->create([
            'user_id' => auth()->id(), 'message' => $request->message,
            'ip_address' => $request->ip(),
        ]);
        $ticket->update(['status' => 'customer_reply', 'last_reply_at' => now()]);
        return back()->with('success', 'Reply sent.');
    }

    public function close(Ticket $ticket)
    {
        abort_if($ticket->user_id !== auth()->id(), 403);
        $ticket->update(['status' => 'closed']);
        return back()->with('success', 'Ticket closed.');
    }
}
