<?php
namespace App\Models;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\SoftDeletes;

class Invoice extends Model
{
    use SoftDeletes;

    protected $fillable = [
        'invoice_number', 'user_id', 'status', 'subtotal', 'tax_rate', 'tax_amount',
        'tax2_rate', 'tax2_amount', 'discount', 'credit_applied', 'total',
        'amount_paid', 'balance', 'currency', 'payment_method',
        'date_issued', 'date_due', 'date_paid', 'notes', 'admin_notes',
        'auto_capture', 'reminder_count', 'last_reminder_at',
    ];

    protected function casts(): array
    {
        return [
            'date_issued' => 'date', 'date_due' => 'date', 'date_paid' => 'date',
            'subtotal' => 'decimal:2', 'total' => 'decimal:2', 'balance' => 'decimal:2',
            'auto_capture' => 'boolean', 'last_reminder_at' => 'datetime',
        ];
    }

    protected static function boot()
    {
        parent::boot();
        static::creating(function ($inv) {
            if (!$inv->invoice_number) {
                $last = static::max('id') ?? 0;
                $inv->invoice_number = 'INV-' . str_pad($last + 1, 6, '0', STR_PAD_LEFT);
            }
        });
    }

    public function user() { return $this->belongsTo(User::class); }
    public function items() { return $this->hasMany(InvoiceItem::class); }
    public function transactions() { return $this->hasMany(Transaction::class); }
    public function order() { return $this->hasOne(Order::class); }

    public function isPaid(): bool { return $this->status === 'paid'; }
    public function isOverdue(): bool { return $this->status !== 'paid' && $this->date_due < now(); }

    public function recalculate(): void
    {
        $this->subtotal = $this->items()->sum('amount');
        $this->tax_amount = round($this->subtotal * ($this->tax_rate / 100), 2);
        $this->total = $this->subtotal + $this->tax_amount + $this->tax2_amount - $this->discount - $this->credit_applied;
        $this->balance = $this->total - $this->amount_paid;
        $this->save();
    }

    public function markPaid(?string $paymentMethod = null): void
    {
        $this->update([
            'status' => 'paid', 'date_paid' => now(), 'balance' => 0,
            'amount_paid' => $this->total,
            'payment_method' => $paymentMethod ?? $this->payment_method,
        ]);
    }

    public function scopeUnpaid($q) { return $q->whereIn('status', ['unpaid', 'overdue']); }
    public function scopeOverdue($q) { return $q->where('status', '!=', 'paid')->where('date_due', '<', now()); }
}
