<?php
namespace App\Models;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\SoftDeletes;
use Illuminate\Support\Str;

class Ticket extends Model
{
    use SoftDeletes;

    protected $fillable = [
        'ticket_number', 'user_id', 'department_id', 'assigned_to', 'service_id',
        'domain_id', 'subject', 'status', 'priority', 'guest_name', 'guest_email',
        'last_reply_at', 'first_response_at', 'flagged', 'admin_notes',
    ];

    protected function casts(): array
    {
        return ['last_reply_at' => 'datetime', 'first_response_at' => 'datetime', 'flagged' => 'boolean'];
    }

    protected static function boot()
    {
        parent::boot();
        static::creating(function ($t) {
            $t->ticket_number = $t->ticket_number ?? 'TKT-' . strtoupper(Str::random(8));
        });
    }

    public function user() { return $this->belongsTo(User::class); }
    public function department() { return $this->belongsTo(Department::class); }
    public function assignedTo() { return $this->belongsTo(User::class, 'assigned_to'); }
    public function service() { return $this->belongsTo(Service::class); }
    public function replies() { return $this->hasMany(TicketReply::class)->orderBy('created_at'); }
    public function attachments() { return $this->hasMany(TicketAttachment::class); }
    public function tags() { return $this->belongsToMany(TicketTag::class, 'ticket_tag_pivot'); }

    public function scopeOpen($q) { return $q->whereNotIn('status', ['closed']); }
    public function scopeByDepartment($q, $id) { return $q->where('department_id', $id); }
}
