<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\SoftDeletes;
use Illuminate\Foundation\Auth\User as Authenticatable;
use Illuminate\Notifications\Notifiable;
use Illuminate\Support\Str;

class User extends Authenticatable
{
    use HasFactory, Notifiable, SoftDeletes;

    protected $fillable = [
        'email', 'password', 'role', 'is_active',
        'first_name', 'last_name', 'company_name',
        'address1', 'address2', 'city', 'state', 'postcode', 'country',
        'phone', 'credit_balance', 'default_currency', 'default_payment_method',
        'tax_id', 'tax_exempt', 'stripe_customer_id', 'language', 'notes',
        'last_login_at', 'last_login_ip',
    ];

    protected $hidden = ['password', 'remember_token', 'two_factor_secret'];

    protected function casts(): array
    {
        return [
            'email_verified_at' => 'datetime',
            'password' => 'hashed',
            'is_active' => 'boolean',
            'tax_exempt' => 'boolean',
            'credit_balance' => 'decimal:2',
            'last_login_at' => 'datetime',
        ];
    }

    protected static function boot()
    {
        parent::boot();
        static::creating(fn($u) => $u->uuid = $u->uuid ?? (string) Str::uuid());
    }

    public function getFullNameAttribute(): string { return trim("{$this->first_name} {$this->last_name}"); }
    public function getDisplayNameAttribute(): string { return $this->company_name ?: $this->full_name; }
    public function isAdmin(): bool { return $this->role === 'admin'; }
    public function isStaff(): bool { return in_array($this->role, ['admin', 'support']); }
    public function isClient(): bool { return $this->role === 'client'; }

    public function services() { return $this->hasMany(Service::class); }
    public function activeServices() { return $this->services()->where('status', 'active'); }
    public function domains() { return $this->hasMany(Domain::class); }
    public function invoices() { return $this->hasMany(Invoice::class); }
    public function unpaidInvoices() { return $this->invoices()->where('status', 'unpaid'); }
    public function orders() { return $this->hasMany(Order::class); }
    public function tickets() { return $this->hasMany(Ticket::class); }
    public function transactions() { return $this->hasMany(Transaction::class); }
    public function paymentMethods() { return $this->hasMany(PaymentMethod::class); }
    public function creditLog() { return $this->hasMany(CreditLog::class); }
    public function affiliate() { return $this->hasOne(Affiliate::class); }

    public function addCredit(float $amount, string $desc, ?int $adminId = null, ?int $invoiceId = null): void
    {
        $this->increment('credit_balance', $amount);
        $this->creditLog()->create([
            'amount' => $amount, 'balance_after' => $this->credit_balance,
            'description' => $desc, 'admin_id' => $adminId, 'invoice_id' => $invoiceId,
        ]);
    }

    public function scopeClients($q) { return $q->where('role', 'client'); }
    public function scopeStaff($q) { return $q->whereIn('role', ['admin', 'support']); }
    public function scopeActive($q) { return $q->where('is_active', true); }
    public function scopeSearch($q, string $term)
    {
        return $q->where(fn($q) => $q->where('first_name', 'like', "%{$term}%")
            ->orWhere('last_name', 'like', "%{$term}%")
            ->orWhere('email', 'like', "%{$term}%")
            ->orWhere('company_name', 'like', "%{$term}%"));
    }
}
