<?php

namespace App\Services;

use App\Models\{Service, Invoice, InvoiceItem, User, AutomationLog, Setting};
use Carbon\Carbon;
use Illuminate\Support\Facades\Log;

class BillingService
{
    /**
     * Generate invoices for services due within X days
     */
    public function generateInvoices(int $daysBefore = 14): int
    {
        $cutoff = Carbon::now()->addDays($daysBefore);
        $count = 0;

        $services = Service::where('status', 'active')
            ->where('billing_cycle', '!=', 'free')
            ->where('next_due_date', '<=', $cutoff)
            ->whereDoesntHave('invoiceItems', function ($q) {
                $q->whereHas('invoice', fn($q) => $q->whereIn('status', ['unpaid', 'partially_paid']));
            })
            ->with(['user', 'product'])
            ->get()
            ->groupBy('user_id');

        foreach ($services as $userId => $userServices) {
            $user = $userServices->first()->user;

            $invoice = Invoice::create([
                'user_id' => $userId,
                'invoice_number' => Invoice::generateNumber(),
                'date_issued' => now(),
                'date_due' => $userServices->first()->next_due_date,
                'tax_rate' => (float) Setting::get('tax_rate', 0),
                'status' => 'unpaid',
            ]);

            foreach ($userServices as $service) {
                $cycleName = ucfirst($service->billing_cycle);
                $invoice->items()->create([
                    'service_id' => $service->id,
                    'description' => "{$service->product->name} - {$service->domain} ({$cycleName})",
                    'amount' => $service->recurring_amount,
                ]);
            }

            $invoice->recalculate();

            // Try to apply client credit
            if ($user->credit > 0 && $invoice->balance > 0) {
                $creditToApply = min($user->credit, $invoice->balance);
                $invoice->applyPayment($creditToApply, 'credit');
                $user->decrement('credit', $creditToApply);
            }

            $count++;
        }

        AutomationLog::create([
            'task' => 'invoice_generate',
            'description' => "Generated {$count} invoices for services due before {$cutoff->format('Y-m-d')}",
            'status' => 'success',
        ]);

        return $count;
    }

    /**
     * Mark overdue invoices
     */
    public function processOverdueInvoices(): int
    {
        $count = Invoice::where('status', 'unpaid')
            ->where('date_due', '<', now())
            ->update(['status' => 'overdue']);

        if ($count > 0) {
            AutomationLog::create([
                'task' => 'overdue_invoices',
                'description' => "Marked {$count} invoices as overdue",
                'status' => 'success',
            ]);
        }

        return $count;
    }

    /**
     * Suspend services with overdue invoices
     */
    public function autoSuspend(): int
    {
        $graceDays = (int) Setting::get('suspension_grace_days', 3);
        $cutoff = Carbon::now()->subDays($graceDays);
        $count = 0;

        $overdueInvoices = Invoice::where('status', 'overdue')
            ->where('date_due', '<', $cutoff)
            ->with('items.service')
            ->get();

        foreach ($overdueInvoices as $invoice) {
            foreach ($invoice->items as $item) {
                if ($item->service && $item->service->status === 'active') {
                    $item->service->suspend('Unpaid invoice #' . $invoice->invoice_number);

                    // Also suspend on server
                    try {
                        app(ProvisioningService::class)->suspendAccount(
                            $item->service,
                            'Unpaid invoice'
                        );
                    } catch (\Exception $e) {
                        Log::warning("Failed to suspend on server: {$e->getMessage()}");
                    }

                    $count++;
                }
            }
        }

        if ($count > 0) {
            AutomationLog::create([
                'task' => 'auto_suspend',
                'description' => "Auto-suspended {$count} services for overdue invoices",
                'status' => 'success',
            ]);
        }

        return $count;
    }

    /**
     * Auto-terminate services suspended for too long
     */
    public function autoTerminate(): int
    {
        $terminateDays = (int) Setting::get('termination_days', 30);
        $cutoff = Carbon::now()->subDays($terminateDays);
        $count = 0;

        $services = Service::where('status', 'suspended')
            ->where('updated_at', '<', $cutoff)
            ->get();

        foreach ($services as $service) {
            try {
                app(ProvisioningService::class)->terminateAccount($service);
            } catch (\Exception $e) {
                Log::warning("Failed to terminate on server: {$e->getMessage()}");
            }

            $service->terminate();
            $count++;
        }

        if ($count > 0) {
            AutomationLog::create([
                'task' => 'auto_terminate',
                'description' => "Auto-terminated {$count} services",
                'status' => 'success',
            ]);
        }

        return $count;
    }

    /**
     * Update next due dates after payment
     */
    public function updateServiceDueDates(Invoice $invoice): void
    {
        foreach ($invoice->items as $item) {
            if (!$item->service) continue;

            $service = $item->service;
            $nextDue = $this->calculateNextDueDate($service->next_due_date, $service->billing_cycle);
            $service->update(['next_due_date' => $nextDue]);
        }
    }

    protected function calculateNextDueDate(Carbon $currentDue, string $cycle): Carbon
    {
        return match($cycle) {
            'monthly' => $currentDue->addMonth(),
            'quarterly' => $currentDue->addMonths(3),
            'semiannually' => $currentDue->addMonths(6),
            'annually' => $currentDue->addYear(),
            'biennially' => $currentDue->addYears(2),
            'triennially' => $currentDue->addYears(3),
            default => $currentDue->addMonth(),
        };
    }

    /**
     * Send payment reminders
     */
    public function sendReminders(): int
    {
        $reminderDays = [1, 3, 7]; // Days before due
        $count = 0;

        foreach ($reminderDays as $days) {
            $date = Carbon::now()->addDays($days)->format('Y-m-d');
            $invoices = Invoice::where('status', 'unpaid')
                ->whereDate('date_due', $date)
                ->with('user')
                ->get();

            foreach ($invoices as $invoice) {
                // Send email notification (integrate with mail service)
                Log::info("Payment reminder sent for invoice #{$invoice->invoice_number} to {$invoice->user->email}");
                $count++;
            }
        }

        return $count;
    }
}
