<?php

namespace App\Services;

use App\Models\{Service, Server, ServerGroup};
use Illuminate\Support\Facades\{Http, Log, Crypt};
use Exception;

class ProvisioningService
{
    /**
     * Create a hosting account on the server
     */
    public function createAccount(Service $service): array
    {
        $server = $this->resolveServer($service);
        $module = $server->module;

        $result = match($module) {
            'cpanel' => $this->cpanelCreateAccount($server, $service),
            'plesk' => $this->pleskCreateAccount($server, $service),
            'directadmin' => $this->directAdminCreateAccount($server, $service),
            default => throw new Exception("Unknown module: {$module}"),
        };

        // Update service with server info
        $service->update([
            'server_id' => $server->id,
            'username' => $result['username'],
            'password' => Crypt::encryptString($result['password']),
            'status' => 'active',
            'module_data' => $result,
        ]);

        Log::info("Account provisioned: {$service->domain} on {$server->hostname}");
        return $result;
    }

    /**
     * Suspend a hosting account
     */
    public function suspendAccount(Service $service, string $reason = ''): bool
    {
        $server = $service->server;
        if (!$server) throw new Exception('No server assigned to this service.');

        return match($server->module) {
            'cpanel' => $this->cpanelSuspendAccount($server, $service, $reason),
            'plesk' => $this->pleskSuspendAccount($server, $service),
            'directadmin' => $this->directAdminSuspendAccount($server, $service, $reason),
            default => throw new Exception("Unknown module: {$server->module}"),
        };
    }

    /**
     * Unsuspend a hosting account
     */
    public function unsuspendAccount(Service $service): bool
    {
        $server = $service->server;
        if (!$server) throw new Exception('No server assigned.');

        return match($server->module) {
            'cpanel' => $this->cpanelUnsuspendAccount($server, $service),
            'plesk' => $this->pleskUnsuspendAccount($server, $service),
            'directadmin' => $this->directAdminUnsuspendAccount($server, $service),
            default => throw new Exception("Unknown module: {$server->module}"),
        };
    }

    /**
     * Terminate/delete a hosting account
     */
    public function terminateAccount(Service $service): bool
    {
        $server = $service->server;
        if (!$server) throw new Exception('No server assigned.');

        return match($server->module) {
            'cpanel' => $this->cpanelTerminateAccount($server, $service),
            'plesk' => $this->pleskTerminateAccount($server, $service),
            'directadmin' => $this->directAdminTerminateAccount($server, $service),
            default => throw new Exception("Unknown module: {$server->module}"),
        };
    }

    /**
     * Test server connection
     */
    public function testConnection(Server $server): string
    {
        return match($server->module) {
            'cpanel' => $this->cpanelTestConnection($server),
            'plesk' => $this->pleskTestConnection($server),
            'directadmin' => $this->directAdminTestConnection($server),
            default => throw new Exception("Unknown module: {$server->module}"),
        };
    }

    /**
     * Resolve which server to use based on server group
     */
    protected function resolveServer(Service $service): Server
    {
        $product = $service->product;

        if (!$product->server_group_id) {
            throw new Exception('No server group configured for this product.');
        }

        $serverGroup = ServerGroup::findOrFail($product->server_group_id);
        $server = $serverGroup->getAvailableServer();

        if (!$server) {
            throw new Exception('No available servers in the server group.');
        }

        return $server;
    }

    protected function generateUsername(string $domain): string
    {
        $base = preg_replace('/[^a-z0-9]/', '', strtolower(explode('.', $domain)[0]));
        return substr($base, 0, 8) . rand(100, 999);
    }

    protected function generatePassword(): string
    {
        return bin2hex(random_bytes(8));
    }

    // ============================================
    // cPanel / WHM API Methods
    // ============================================
    protected function cpanelApiUrl(Server $server): string
    {
        $protocol = $server->use_ssl ? 'https' : 'http';
        return "{$protocol}://{$server->hostname}:{$server->port}";
    }

    protected function cpanelApiRequest(Server $server, string $endpoint, array $params = []): array
    {
        $url = $this->cpanelApiUrl($server) . $endpoint;
        $response = Http::withHeaders([
            'Authorization' => 'WHM ' . $server->username . ':' . Crypt::decryptString($server->access_hash),
        ])->timeout(30)->get($url, $params);

        if (!$response->successful()) {
            throw new Exception("cPanel API error: " . $response->body());
        }

        return $response->json();
    }

    protected function cpanelCreateAccount(Server $server, Service $service): array
    {
        $username = $this->generateUsername($service->domain);
        $password = $this->generatePassword();
        $package = $service->product->module_config['package'] ?? 'default';

        $result = $this->cpanelApiRequest($server, '/json-api/createacct', [
            'api.version' => 1,
            'username' => $username,
            'domain' => $service->domain,
            'password' => $password,
            'plan' => $package,
            'contactemail' => $service->user->email,
        ]);

        if (($result['metadata']['result'] ?? 0) !== 1) {
            throw new Exception('cPanel account creation failed: ' . ($result['metadata']['reason'] ?? 'Unknown error'));
        }

        return [
            'username' => $username,
            'password' => $password,
            'package' => $package,
            'server' => $server->hostname,
            'nameservers' => [$server->nameserver1, $server->nameserver2],
        ];
    }

    protected function cpanelSuspendAccount(Server $server, Service $service, string $reason): bool
    {
        $result = $this->cpanelApiRequest($server, '/json-api/suspendacct', [
            'api.version' => 1,
            'user' => $service->username,
            'reason' => $reason,
        ]);
        return ($result['metadata']['result'] ?? 0) === 1;
    }

    protected function cpanelUnsuspendAccount(Server $server, Service $service): bool
    {
        $result = $this->cpanelApiRequest($server, '/json-api/unsuspendacct', [
            'api.version' => 1,
            'user' => $service->username,
        ]);
        return ($result['metadata']['result'] ?? 0) === 1;
    }

    protected function cpanelTerminateAccount(Server $server, Service $service): bool
    {
        $result = $this->cpanelApiRequest($server, '/json-api/removeacct', [
            'api.version' => 1,
            'username' => $service->username,
        ]);
        return ($result['metadata']['result'] ?? 0) === 1;
    }

    protected function cpanelTestConnection(Server $server): string
    {
        $result = $this->cpanelApiRequest($server, '/json-api/version');
        return "WHM Version: " . ($result['version'] ?? 'Connected');
    }

    // ============================================
    // Plesk API Methods
    // ============================================
    protected function pleskApiRequest(Server $server, string $xml): string
    {
        $protocol = $server->use_ssl ? 'https' : 'http';
        $port = $server->port ?: 8443;
        $url = "{$protocol}://{$server->hostname}:{$port}/enterprise/control/agent.php";

        $response = Http::withHeaders([
            'Content-Type' => 'text/xml',
            'HTTP_AUTH_LOGIN' => $server->username,
            'HTTP_AUTH_PASSWD' => Crypt::decryptString($server->password),
        ])->timeout(30)->withBody($xml, 'text/xml')->post($url);

        if (!$response->successful()) {
            throw new Exception('Plesk API error: ' . $response->body());
        }

        return $response->body();
    }

    protected function pleskCreateAccount(Server $server, Service $service): array
    {
        $username = $this->generateUsername($service->domain);
        $password = $this->generatePassword();
        $plan = $service->product->module_config['plan'] ?? 'Default Domain';

        $xml = <<<XML
        <packet>
            <webspace><add>
                <gen_setup>
                    <name>{$service->domain}</name>
                    <owner-login>{$server->username}</owner-login>
                    <htype>vrt_hst</htype>
                    <ip_address>{$server->ip_address}</ip_address>
                </gen_setup>
                <hosting><vrt_hst>
                    <property><name>ftp_login</name><value>{$username}</value></property>
                    <property><name>ftp_password</name><value>{$password}</value></property>
                    <ip_address>{$server->ip_address}</ip_address>
                </vrt_hst></hosting>
                <plan-name>{$plan}</plan-name>
            </add></webspace>
        </packet>
        XML;

        $this->pleskApiRequest($server, $xml);

        return [
            'username' => $username,
            'password' => $password,
            'plan' => $plan,
            'server' => $server->hostname,
        ];
    }

    protected function pleskSuspendAccount(Server $server, Service $service): bool
    {
        $xml = "<packet><webspace><set><filter><name>{$service->domain}</name></filter><values><gen_setup><status>16</status></gen_setup></values></set></webspace></packet>";
        $this->pleskApiRequest($server, $xml);
        return true;
    }

    protected function pleskUnsuspendAccount(Server $server, Service $service): bool
    {
        $xml = "<packet><webspace><set><filter><name>{$service->domain}</name></filter><values><gen_setup><status>0</status></gen_setup></values></set></webspace></packet>";
        $this->pleskApiRequest($server, $xml);
        return true;
    }

    protected function pleskTerminateAccount(Server $server, Service $service): bool
    {
        $xml = "<packet><webspace><del><filter><name>{$service->domain}</name></filter></del></webspace></packet>";
        $this->pleskApiRequest($server, $xml);
        return true;
    }

    protected function pleskTestConnection(Server $server): string
    {
        $xml = "<packet><server><get><gen/></get></server></packet>";
        $this->pleskApiRequest($server, $xml);
        return "Plesk connection successful";
    }

    // ============================================
    // DirectAdmin API Methods
    // ============================================
    protected function directAdminApiRequest(Server $server, string $endpoint, array $params = [], string $method = 'GET'): string
    {
        $protocol = $server->use_ssl ? 'https' : 'http';
        $port = $server->port ?: 2222;
        $url = "{$protocol}://{$server->hostname}:{$port}{$endpoint}";

        $response = Http::withBasicAuth($server->username, Crypt::decryptString($server->password))
            ->timeout(30);

        $response = $method === 'POST'
            ? $response->asForm()->post($url, $params)
            : $response->get($url, $params);

        if (!$response->successful()) {
            throw new Exception('DirectAdmin API error: ' . $response->body());
        }

        return $response->body();
    }

    protected function directAdminCreateAccount(Server $server, Service $service): array
    {
        $username = $this->generateUsername($service->domain);
        $password = $this->generatePassword();
        $package = $service->product->module_config['package'] ?? 'default';

        $this->directAdminApiRequest($server, '/CMD_API_ACCOUNT_USER', [
            'action' => 'create',
            'add' => 'Submit',
            'username' => $username,
            'email' => $service->user->email,
            'passwd' => $password,
            'passwd2' => $password,
            'domain' => $service->domain,
            'package' => $package,
            'ip' => $server->ip_address,
        ], 'POST');

        return [
            'username' => $username,
            'password' => $password,
            'package' => $package,
            'server' => $server->hostname,
        ];
    }

    protected function directAdminSuspendAccount(Server $server, Service $service, string $reason): bool
    {
        $this->directAdminApiRequest($server, '/CMD_API_SELECT_USERS', [
            'location' => 'CMD_SELECT_USERS',
            'suspend' => 'Suspend',
            'select0' => $service->username,
        ], 'POST');
        return true;
    }

    protected function directAdminUnsuspendAccount(Server $server, Service $service): bool
    {
        $this->directAdminApiRequest($server, '/CMD_API_SELECT_USERS', [
            'location' => 'CMD_SELECT_USERS',
            'unsuspend' => 'Unsuspend',
            'select0' => $service->username,
        ], 'POST');
        return true;
    }

    protected function directAdminTerminateAccount(Server $server, Service $service): bool
    {
        $this->directAdminApiRequest($server, '/CMD_API_SELECT_USERS', [
            'confirmed' => 'Confirm',
            'delete' => 'yes',
            'select0' => $service->username,
        ], 'POST');
        return true;
    }

    protected function directAdminTestConnection(Server $server): string
    {
        $result = $this->directAdminApiRequest($server, '/CMD_API_SHOW_ALL_USERS');
        return "DirectAdmin connection successful";
    }
}
