<?php

use Illuminate\Database\Migrations\Migration;
use Illuminate\Database\Schema\Blueprint;
use Illuminate\Support\Facades\Schema;

return new class extends Migration
{
    public function up(): void
    {
        // Server Groups (must exist before products reference them)
        Schema::create('server_groups', function (Blueprint $table) {
            $table->id();
            $table->string('name', 200);
            $table->enum('fill_type', ['fill', 'round_robin'])->default('fill');
            $table->timestamps();
        });

        // Servers
        Schema::create('servers', function (Blueprint $table) {
            $table->id();
            $table->unsignedBigInteger('server_group_id')->nullable();
            $table->string('name', 200);
            $table->string('hostname', 255);
            $table->string('ip_address', 45);
            $table->json('assigned_ips')->nullable();
            $table->string('nameserver1', 255)->nullable();
            $table->string('nameserver2', 255)->nullable();
            $table->string('nameserver3', 255)->nullable();
            $table->string('nameserver4', 255)->nullable();
            $table->string('module', 100);
            $table->string('username', 100)->nullable();
            $table->text('password')->nullable();
            $table->text('access_hash')->nullable();
            $table->integer('port')->default(2087);
            $table->boolean('use_ssl')->default(true);
            $table->integer('max_accounts')->default(0);
            $table->integer('active_accounts')->default(0);
            $table->boolean('is_active')->default(true);
            $table->boolean('is_disabled')->default(false);
            $table->integer('sort_order')->default(0);
            $table->timestamps();

            $table->foreign('server_group_id')->references('id')->on('server_groups')->nullOnDelete();
            $table->index('module');
        });

        // Product Groups
        Schema::create('product_groups', function (Blueprint $table) {
            $table->id();
            $table->string('name', 200);
            $table->string('slug', 200)->unique();
            $table->text('description')->nullable();
            $table->string('headline', 255)->nullable();
            $table->integer('sort_order')->default(0);
            $table->boolean('is_hidden')->default(false);
            $table->timestamps();
        });

        // Products
        Schema::create('products', function (Blueprint $table) {
            $table->id();
            $table->unsignedBigInteger('product_group_id');
            $table->string('name', 200);
            $table->string('slug', 200)->unique();
            $table->enum('type', ['hosting', 'reseller', 'vps', 'dedicated', 'domain', 'addon', 'other'])->default('hosting');
            $table->text('description')->nullable();
            $table->json('features')->nullable();
            $table->string('module', 100)->nullable();
            $table->json('module_config')->nullable();
            $table->integer('stock_quantity')->default(-1);
            $table->boolean('stock_enabled')->default(false);
            $table->boolean('is_active')->default(true);
            $table->boolean('is_hidden')->default(false);
            $table->boolean('auto_setup')->default(true);
            $table->enum('auto_setup_on', ['payment', 'order', 'manual'])->default('payment');
            $table->boolean('require_domain')->default(false);
            $table->json('upgrade_product_ids')->nullable();
            $table->json('configurable_options')->nullable();
            $table->string('welcome_email_template', 100)->nullable();
            $table->integer('sort_order')->default(0);
            $table->timestamps();
            $table->softDeletes();

            $table->foreign('product_group_id')->references('id')->on('product_groups')->cascadeOnDelete();
            $table->index(['product_group_id', 'is_active']);
            $table->index('type');
        });

        // Product Pricing
        Schema::create('product_pricing', function (Blueprint $table) {
            $table->id();
            $table->unsignedBigInteger('product_id');
            $table->string('currency', 3)->default('USD');
            $table->enum('billing_cycle', [
                'free', 'one_time', 'monthly', 'quarterly',
                'semi_annually', 'annually', 'biennially', 'triennially'
            ])->default('monthly');
            $table->decimal('price', 12, 2);
            $table->decimal('setup_fee', 12, 2)->default(0);
            $table->boolean('is_enabled')->default(true);
            $table->timestamps();

            $table->foreign('product_id')->references('id')->on('products')->cascadeOnDelete();
            $table->unique(['product_id', 'currency', 'billing_cycle']);
        });

        // Pivot: products <-> server_groups
        Schema::create('product_server_group', function (Blueprint $table) {
            $table->unsignedBigInteger('product_id');
            $table->unsignedBigInteger('server_group_id');
            $table->primary(['product_id', 'server_group_id']);
            $table->foreign('product_id')->references('id')->on('products')->cascadeOnDelete();
            $table->foreign('server_group_id')->references('id')->on('server_groups')->cascadeOnDelete();
        });

        // Configurable Option Groups
        Schema::create('config_option_groups', function (Blueprint $table) {
            $table->id();
            $table->string('name', 200);
            $table->text('description')->nullable();
            $table->timestamps();
        });

        Schema::create('config_options', function (Blueprint $table) {
            $table->id();
            $table->unsignedBigInteger('config_option_group_id');
            $table->string('name', 200);
            $table->enum('type', ['dropdown', 'radio', 'checkbox', 'quantity'])->default('dropdown');
            $table->integer('sort_order')->default(0);
            $table->timestamps();
            $table->foreign('config_option_group_id')->references('id')->on('config_option_groups')->cascadeOnDelete();
        });

        Schema::create('config_option_values', function (Blueprint $table) {
            $table->id();
            $table->unsignedBigInteger('config_option_id');
            $table->string('name', 200);
            $table->decimal('monthly_price', 12, 2)->default(0);
            $table->decimal('quarterly_price', 12, 2)->default(0);
            $table->decimal('semi_annual_price', 12, 2)->default(0);
            $table->decimal('annual_price', 12, 2)->default(0);
            $table->decimal('biennial_price', 12, 2)->default(0);
            $table->decimal('setup_fee', 12, 2)->default(0);
            $table->integer('sort_order')->default(0);
            $table->timestamps();
            $table->foreign('config_option_id')->references('id')->on('config_options')->cascadeOnDelete();
        });

        Schema::create('product_config_group', function (Blueprint $table) {
            $table->unsignedBigInteger('product_id');
            $table->unsignedBigInteger('config_option_group_id');
            $table->primary(['product_id', 'config_option_group_id']);
            $table->foreign('product_id')->references('id')->on('products')->cascadeOnDelete();
            $table->foreign('config_option_group_id')->references('id')->on('config_option_groups')->cascadeOnDelete();
        });
    }

    public function down(): void
    {
        Schema::dropIfExists('product_config_group');
        Schema::dropIfExists('config_option_values');
        Schema::dropIfExists('config_options');
        Schema::dropIfExists('config_option_groups');
        Schema::dropIfExists('product_server_group');
        Schema::dropIfExists('product_pricing');
        Schema::dropIfExists('products');
        Schema::dropIfExists('product_groups');
        Schema::dropIfExists('servers');
        Schema::dropIfExists('server_groups');
    }
};
