<?php

use Illuminate\Database\Migrations\Migration;
use Illuminate\Database\Schema\Blueprint;
use Illuminate\Support\Facades\Schema;

return new class extends Migration
{
    public function up(): void
    {
        // Currencies (no FK)
        Schema::create('currencies', function (Blueprint $table) {
            $table->id();
            $table->string('code', 3)->unique();
            $table->string('name', 100);
            $table->string('symbol', 10);
            $table->string('symbol_position', 10)->default('left');
            $table->string('decimal_separator', 1)->default('.');
            $table->string('thousand_separator', 1)->default(',');
            $table->integer('decimals')->default(2);
            $table->decimal('exchange_rate', 12, 6)->default(1);
            $table->boolean('is_default')->default(false);
            $table->timestamps();
        });

        // Tax Rules (no FK)
        Schema::create('tax_rules', function (Blueprint $table) {
            $table->id();
            $table->string('name', 100);
            $table->decimal('rate', 5, 2);
            $table->string('country', 2)->default('*');
            $table->string('state', 100)->default('*');
            $table->boolean('is_inclusive')->default(false);
            $table->integer('priority')->default(1);
            $table->boolean('is_active')->default(true);
            $table->timestamps();
            $table->index(['country', 'state']);
        });

        // Promotions (no FK)
        Schema::create('promotions', function (Blueprint $table) {
            $table->id();
            $table->string('code', 50)->unique();
            $table->string('name', 200);
            $table->enum('type', ['percentage', 'fixed', 'override', 'free_setup'])->default('percentage');
            $table->decimal('value', 12, 2);
            $table->json('applies_to_products')->nullable();
            $table->json('applies_to_billing_cycles')->nullable();
            $table->boolean('recurring')->default(false);
            $table->integer('recurring_limit')->default(0);
            $table->integer('max_uses')->default(0);
            $table->integer('uses_count')->default(0);
            $table->integer('max_uses_per_client')->default(0);
            $table->decimal('min_order_amount', 12, 2)->default(0);
            $table->boolean('requires_existing_service')->default(false);
            $table->date('start_date')->nullable();
            $table->date('end_date')->nullable();
            $table->boolean('is_active')->default(true);
            $table->text('notes')->nullable();
            $table->timestamps();
        });

        // Invoices (refs: users)
        Schema::create('invoices', function (Blueprint $table) {
            $table->id();
            $table->string('invoice_number', 30)->unique();
            $table->unsignedBigInteger('user_id');
            $table->enum('status', ['draft', 'unpaid', 'paid', 'cancelled', 'refunded', 'collections', 'overdue'])->default('unpaid');
            $table->decimal('subtotal', 12, 2)->default(0);
            $table->decimal('tax_rate', 5, 2)->default(0);
            $table->decimal('tax_amount', 12, 2)->default(0);
            $table->decimal('tax2_rate', 5, 2)->default(0);
            $table->decimal('tax2_amount', 12, 2)->default(0);
            $table->decimal('discount', 12, 2)->default(0);
            $table->decimal('credit_applied', 12, 2)->default(0);
            $table->decimal('total', 12, 2)->default(0);
            $table->decimal('amount_paid', 12, 2)->default(0);
            $table->decimal('balance', 12, 2)->default(0);
            $table->string('currency', 3)->default('USD');
            $table->string('payment_method', 50)->nullable();
            $table->date('date_issued')->nullable();
            $table->date('date_due')->nullable();
            $table->date('date_paid')->nullable();
            $table->text('notes')->nullable();
            $table->text('admin_notes')->nullable();
            $table->boolean('auto_capture')->default(true);
            $table->integer('reminder_count')->default(0);
            $table->timestamp('last_reminder_at')->nullable();
            $table->timestamps();
            $table->softDeletes();

            $table->foreign('user_id')->references('id')->on('users')->cascadeOnDelete();
            $table->index(['user_id', 'status']);
            $table->index(['status', 'date_due']);
        });

        // Orders (refs: users, invoices)
        Schema::create('orders', function (Blueprint $table) {
            $table->id();
            $table->string('order_number', 20)->unique();
            $table->unsignedBigInteger('user_id');
            $table->unsignedBigInteger('invoice_id')->nullable();
            $table->enum('status', ['pending', 'active', 'fraud', 'cancelled'])->default('pending');
            $table->decimal('amount', 12, 2)->default(0);
            $table->string('currency', 3)->default('USD');
            $table->string('payment_method', 50)->nullable();
            $table->string('ip_address', 45)->nullable();
            $table->text('notes')->nullable();
            $table->text('fraud_output')->nullable();
            $table->string('promo_code', 50)->nullable();
            $table->timestamps();

            $table->foreign('user_id')->references('id')->on('users')->cascadeOnDelete();
            $table->foreign('invoice_id')->references('id')->on('invoices')->nullOnDelete();
            $table->index(['user_id', 'status']);
        });

        // Services (refs: users, orders, products, servers)
        Schema::create('services', function (Blueprint $table) {
            $table->id();
            $table->unsignedBigInteger('user_id');
            $table->unsignedBigInteger('order_id')->nullable();
            $table->unsignedBigInteger('product_id');
            $table->unsignedBigInteger('server_id')->nullable();
            $table->string('domain', 255)->nullable();
            $table->enum('status', ['pending', 'active', 'suspended', 'terminated', 'cancelled', 'fraud', 'completed'])->default('pending');
            $table->string('suspension_reason', 255)->nullable();
            $table->string('username', 100)->nullable();
            $table->text('password')->nullable();
            $table->string('dedicated_ip', 45)->nullable();
            $table->string('assigned_ips', 500)->nullable();
            $table->enum('billing_cycle', ['free', 'one_time', 'monthly', 'quarterly', 'semi_annually', 'annually', 'biennially', 'triennially'])->default('monthly');
            $table->decimal('recurring_amount', 12, 2)->default(0);
            $table->string('currency', 3)->default('USD');
            $table->date('registration_date')->nullable();
            $table->date('next_due_date')->nullable();
            $table->date('termination_date')->nullable();
            $table->json('module_data')->nullable();
            $table->bigInteger('disk_usage')->default(0);
            $table->bigInteger('disk_limit')->default(0);
            $table->bigInteger('bandwidth_usage')->default(0);
            $table->bigInteger('bandwidth_limit')->default(0);
            $table->text('admin_notes')->nullable();
            $table->timestamps();
            $table->softDeletes();

            $table->foreign('user_id')->references('id')->on('users')->cascadeOnDelete();
            $table->foreign('order_id')->references('id')->on('orders')->nullOnDelete();
            $table->foreign('product_id')->references('id')->on('products')->restrictOnDelete();
            $table->foreign('server_id')->references('id')->on('servers')->nullOnDelete();
            $table->index(['user_id', 'status']);
            $table->index(['status', 'next_due_date']);
            $table->index('domain');
        });

        // Service Addons (refs: services, products)
        Schema::create('service_addons', function (Blueprint $table) {
            $table->id();
            $table->unsignedBigInteger('service_id');
            $table->unsignedBigInteger('product_id');
            $table->string('name', 200);
            $table->decimal('recurring_amount', 12, 2)->default(0);
            $table->decimal('setup_fee', 12, 2)->default(0);
            $table->enum('billing_cycle', ['free', 'one_time', 'monthly', 'quarterly', 'semi_annually', 'annually', 'biennially', 'triennially'])->default('monthly');
            $table->enum('status', ['pending', 'active', 'suspended', 'terminated', 'cancelled'])->default('pending');
            $table->date('next_due_date')->nullable();
            $table->timestamps();

            $table->foreign('service_id')->references('id')->on('services')->cascadeOnDelete();
            $table->foreign('product_id')->references('id')->on('products')->restrictOnDelete();
        });

        // Invoice Items (refs: invoices, services)
        Schema::create('invoice_items', function (Blueprint $table) {
            $table->id();
            $table->unsignedBigInteger('invoice_id');
            $table->unsignedBigInteger('service_id')->nullable();
            $table->unsignedBigInteger('domain_id')->nullable(); // FK added later after domains table
            $table->enum('type', ['hosting', 'domain_register', 'domain_renew', 'domain_transfer', 'addon', 'setup', 'promo', 'late_fee', 'credit', 'other'])->default('other');
            $table->string('description', 500);
            $table->decimal('amount', 12, 2);
            $table->decimal('tax_amount', 12, 2)->default(0);
            $table->boolean('is_taxed')->default(true);
            $table->date('period_start')->nullable();
            $table->date('period_end')->nullable();
            $table->timestamps();

            $table->foreign('invoice_id')->references('id')->on('invoices')->cascadeOnDelete();
            $table->foreign('service_id')->references('id')->on('services')->nullOnDelete();
        });

        // Transactions (refs: users, invoices)
        Schema::create('transactions', function (Blueprint $table) {
            $table->id();
            $table->unsignedBigInteger('user_id');
            $table->unsignedBigInteger('invoice_id')->nullable();
            $table->string('transaction_id', 255)->nullable();
            $table->string('gateway', 50);
            $table->enum('type', ['payment', 'refund', 'chargeback', 'credit_add', 'credit_remove']);
            $table->decimal('amount', 12, 2);
            $table->decimal('fee', 12, 2)->default(0);
            $table->string('currency', 3)->default('USD');
            $table->enum('status', ['pending', 'completed', 'failed', 'refunded', 'disputed']);
            $table->text('description')->nullable();
            $table->json('gateway_response')->nullable();
            $table->timestamps();

            $table->foreign('user_id')->references('id')->on('users')->cascadeOnDelete();
            $table->foreign('invoice_id')->references('id')->on('invoices')->nullOnDelete();
            $table->index(['user_id', 'created_at']);
            $table->index(['gateway', 'transaction_id']);
        });

        // Payment Methods (refs: users)
        Schema::create('payment_methods', function (Blueprint $table) {
            $table->id();
            $table->unsignedBigInteger('user_id');
            $table->string('gateway', 50);
            $table->string('gateway_token', 255);
            $table->string('type', 30)->nullable();
            $table->string('last_four', 4)->nullable();
            $table->string('expiry_month', 2)->nullable();
            $table->string('expiry_year', 4)->nullable();
            $table->string('card_brand', 20)->nullable();
            $table->boolean('is_default')->default(false);
            $table->timestamps();

            $table->foreign('user_id')->references('id')->on('users')->cascadeOnDelete();
        });

        // Credit Log (refs: users)
        Schema::create('credit_log', function (Blueprint $table) {
            $table->id();
            $table->unsignedBigInteger('user_id');
            $table->unsignedBigInteger('admin_id')->nullable();
            $table->decimal('amount', 12, 2);
            $table->decimal('balance_after', 12, 2);
            $table->string('description', 500);
            $table->unsignedBigInteger('invoice_id')->nullable();
            $table->timestamps();

            $table->foreign('user_id')->references('id')->on('users')->cascadeOnDelete();
            $table->foreign('admin_id')->references('id')->on('users')->nullOnDelete();
            $table->foreign('invoice_id')->references('id')->on('invoices')->nullOnDelete();
        });
    }

    public function down(): void
    {
        Schema::dropIfExists('credit_log');
        Schema::dropIfExists('payment_methods');
        Schema::dropIfExists('transactions');
        Schema::dropIfExists('invoice_items');
        Schema::dropIfExists('service_addons');
        Schema::dropIfExists('services');
        Schema::dropIfExists('orders');
        Schema::dropIfExists('invoices');
        Schema::dropIfExists('promotions');
        Schema::dropIfExists('tax_rules');
        Schema::dropIfExists('currencies');
    }
};
