<?php

use Illuminate\Database\Migrations\Migration;
use Illuminate\Database\Schema\Blueprint;
use Illuminate\Support\Facades\Schema;

return new class extends Migration
{
    public function up(): void
    {
        Schema::create('domain_registrars', function (Blueprint $table) {
            $table->id();
            $table->string('name', 100);
            $table->string('module', 100);
            $table->json('config')->nullable();
            $table->boolean('is_active')->default(true);
            $table->integer('sort_order')->default(0);
            $table->timestamps();
        });

        Schema::create('tld_pricing', function (Blueprint $table) {
            $table->id();
            $table->string('tld', 20);
            $table->unsignedBigInteger('domain_registrar_id');
            $table->string('currency', 3)->default('USD');
            $table->decimal('register_1yr', 12, 2)->nullable();
            $table->decimal('register_2yr', 12, 2)->nullable();
            $table->decimal('register_3yr', 12, 2)->nullable();
            $table->decimal('register_5yr', 12, 2)->nullable();
            $table->decimal('register_10yr', 12, 2)->nullable();
            $table->decimal('transfer_price', 12, 2)->nullable();
            $table->decimal('renew_1yr', 12, 2)->nullable();
            $table->decimal('renew_2yr', 12, 2)->nullable();
            $table->decimal('renew_3yr', 12, 2)->nullable();
            $table->decimal('renew_5yr', 12, 2)->nullable();
            $table->decimal('renew_10yr', 12, 2)->nullable();
            $table->decimal('grace_fee', 12, 2)->default(0);
            $table->decimal('redemption_fee', 12, 2)->default(0);
            $table->decimal('id_protection_price', 12, 2)->default(0);
            $table->boolean('id_protection_available')->default(false);
            $table->boolean('epp_required')->default(true);
            $table->integer('min_years')->default(1);
            $table->integer('max_years')->default(10);
            $table->boolean('is_active')->default(true);
            $table->integer('sort_order')->default(0);
            $table->timestamps();

            $table->foreign('domain_registrar_id')->references('id')->on('domain_registrars')->cascadeOnDelete();
            $table->unique(['tld', 'domain_registrar_id', 'currency']);
        });

        Schema::create('domains', function (Blueprint $table) {
            $table->id();
            $table->unsignedBigInteger('user_id');
            $table->unsignedBigInteger('order_id')->nullable();
            $table->unsignedBigInteger('domain_registrar_id')->nullable();
            $table->string('domain', 255);
            $table->string('sld', 200)->nullable();
            $table->string('tld', 20)->nullable();
            $table->enum('type', ['register', 'transfer', 'existing'])->default('register');
            $table->enum('status', ['pending', 'active', 'expired', 'cancelled', 'transferred_away', 'fraud'])->default('pending');
            $table->date('registration_date')->nullable();
            $table->date('expiry_date')->nullable();
            $table->date('next_due_date')->nullable();
            $table->integer('registration_period')->default(1);
            $table->decimal('recurring_amount', 12, 2)->default(0);
            $table->string('currency', 3)->default('USD');
            $table->boolean('auto_renew')->default(true);
            $table->string('nameserver1', 255)->nullable();
            $table->string('nameserver2', 255)->nullable();
            $table->string('nameserver3', 255)->nullable();
            $table->string('nameserver4', 255)->nullable();
            $table->boolean('id_protection')->default(false);
            $table->text('epp_code')->nullable();
            $table->boolean('dns_management')->default(false);
            $table->boolean('email_forwarding')->default(false);
            $table->text('admin_notes')->nullable();
            $table->timestamps();
            $table->softDeletes();

            $table->foreign('user_id')->references('id')->on('users')->cascadeOnDelete();
            $table->foreign('order_id')->references('id')->on('orders')->nullOnDelete();
            $table->foreign('domain_registrar_id')->references('id')->on('domain_registrars')->nullOnDelete();
            $table->index(['user_id', 'status']);
            $table->index('domain');
        });

        Schema::create('domain_dns_records', function (Blueprint $table) {
            $table->id();
            $table->unsignedBigInteger('domain_id');
            $table->string('name', 255);
            $table->enum('type', ['A', 'AAAA', 'CNAME', 'MX', 'TXT', 'NS', 'SRV', 'CAA']);
            $table->string('content', 500);
            $table->integer('ttl')->default(3600);
            $table->integer('priority')->nullable();
            $table->timestamps();
            $table->foreign('domain_id')->references('id')->on('domains')->cascadeOnDelete();
        });

        Schema::create('domain_contacts', function (Blueprint $table) {
            $table->id();
            $table->unsignedBigInteger('domain_id');
            $table->enum('type', ['registrant', 'admin', 'tech', 'billing']);
            $table->string('first_name', 100);
            $table->string('last_name', 100);
            $table->string('company', 200)->nullable();
            $table->string('email', 255);
            $table->string('phone', 30);
            $table->string('address1', 255);
            $table->string('address2', 255)->nullable();
            $table->string('city', 100);
            $table->string('state', 100);
            $table->string('postcode', 20);
            $table->string('country', 2);
            $table->timestamps();
            $table->foreign('domain_id')->references('id')->on('domains')->cascadeOnDelete();
            $table->unique(['domain_id', 'type']);
        });
    }

    public function down(): void
    {
        Schema::dropIfExists('domain_contacts');
        Schema::dropIfExists('domain_dns_records');
        Schema::dropIfExists('domains');
        Schema::dropIfExists('tld_pricing');
        Schema::dropIfExists('domain_registrars');
    }
};
